<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

class UsuarioController extends Controller
{
    public function index()
    {
        $usuarios = DB::table('users')->paginate(10);

        // Totals
        $total = DB::table('users')->count();
        $activos = DB::table('users')->where('status', 'Activo')->count();
        $inactivos = DB::table('users')->where('status', 'Inactivo')->count();

        return view('usuarios.index', compact('usuarios', 'total', 'activos', 'inactivos'));
    }

    // ✅ Create user
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:6',
            'status' => 'required|string',
        ]);

        $validated['password'] = Hash::make($validated['password']);

        DB::table('users')->insert($validated);

        return response()->json(['success' => true]);
    }

    // ✅ Update user
    public function update(Request $request, $id)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $id,
            'status' => 'required|string',
        ]);

        DB::table('users')->where('id', $id)->update($validated);

        return response()->json(['success' => true]);
    }
    
    // ✅ View logged-in user's profile
public function profile(Request $request)
{
    $user = $request->user(); // authenticated user
    return view('usuarios.profile', compact('user'));
}

// ✅ Update name/email
public function updateProfile(Request $request)
{
    $user = $request->user();

    $validated = $request->validate([
        'name'  => 'required|string|max:255',
        'email' => 'required|email|unique:users,email,' . $user->id,
    ]);

    DB::table('users')
        ->where('id', $user->id)
        ->update([
            'name'  => $validated['name'],
            'email' => $validated['email'],
        ]);

    return back()->with('success', 'Perfil actualizado correctamente.');
}

// ✅ Change password
public function updatePassword(Request $request)
{
    $user = $request->user();

    $request->validate([
        'current_password' => 'required',
        'new_password'     => 'required|min:6|confirmed',
    ]);

    $currentPassword = $request->input('current_password');
    if (!Hash::check($currentPassword, $user->password)) {
        return back()->withErrors(['current_password' => 'La contraseña actual no es correcta.']);
    }

    DB::table('users')
        ->where('id', $user->id)
        ->update(['password' => Hash::make($request->input('new_password'))]);

    return back()->with('success', 'Contraseña actualizada correctamente.');
}


}
