<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;


class TractorController extends Controller
{
public function index()
{
    $tractores = DB::table('tractores as t')
    ->leftJoin('conductores as c', 't.conductor_id', '=', 'c.id')
    ->select(
        't.id',
        't.unidad',
        't.brand',
        't.make',
        't.plate',
        't.serie',
        't.status',
        't.conductor_id', // 👈 Asegura que lo traemos
        't.remolques_ids',
        't.dolly_id',
        DB::raw("CONCAT(c.first_name, ' ', c.last_name) as conductor_name")
    )
    ->orderBy('t.id', 'desc')
    ->paginate(10);


    $total = DB::table('tractores')->count();

    $por_estado = DB::table('tractores')
        ->select('status', DB::raw('COUNT(*) as total'))
        ->groupBy('status')
        ->get()
        ->pluck('total', 'status');

    $conductores = DB::table('conductores')->orderBy('first_name')->get();

    // ✅ Remolques disponibles
    $remolquesDisponibles = DB::table('remolques')
        ->whereIn('status', ['Activo', 'Inactivo'])
        ->orderBy('unidad')
        ->get();

    // ✅ Dollys disponibles
    $dollysDisponibles = DB::table('dollys')
        ->whereIn('status', ['Activo', 'Inactivo'])
        ->orderBy('unidad')
        ->get();

    return view('tractores.index', compact(
        'tractores',
        'total',
        'por_estado',
        'conductores',
        'remolquesDisponibles',
        'dollysDisponibles'
    ));
}



    public function store(Request $request)
    {
        $validated = $request->validate([
            'brand' => 'required|string|max:100',
            'make' => 'required|string|max:100',
            'plate' => 'required|string|max:50|unique:tractores,plate',
            'status' => 'required|string|max:50',
            'conductor_id' => 'nullable|integer|exists:conductores,id',
        ]);

        DB::table('tractores')->insert($validated);

        return response()->json(['success' => true]);
    }

    public function update(Request $request, $id)
    {
        $validated = $request->validate([
            'brand' => 'required|string|max:100',
            'make' => 'required|string|max:100',
            'plate' => 'required|string|max:50|unique:tractores,plate,' . $id,
            'status' => 'required|string|max:50',
            'conductor_id' => 'nullable|integer|exists:conductores,id',
        ]);

        DB::table('tractores')->where('id', $id)->update($validated);

        // If a conductor is assigned, update their status
        if (!empty($validated['conductor_id'])) {
            DB::table('conductores')->where('id', $validated['conductor_id'])->update(['status' => 'Activo']);

        }

        return response()->json(['success' => true]);
    }
    
    


public function assign(Request $request, $id)
{
    $remolques = $request->input('remolques', []);
    $dolly_id = $request->input('dolly_id');
    $conductor_id = $request->input('conductor_id');

    // ✅ Update tractor with conductor if provided
    DB::table('tractores')->where('id', $id)->update([
        'conductor_id' => $conductor_id,
    ]);

    // ✅ If a conductor is assigned, update their status
    if (!empty($conductor_id)) {
        DB::table('conductores')->where('id', $conductor_id)->update(['status' => 'Activo']);
    }

    // ✅ Clear existing assignments
    DB::table('remolques')->where('tractor_id', $id)->update(['tractor_id' => null, 'status' => 'Activo']);
    DB::table('dollys')->where('tractor_id', $id)->update(['tractor_id' => null, 'status' => 'Activo']);

    // ✅ Assign selected remolques
    if (!empty($remolques)) {
        DB::table('remolques')
            ->whereIn('id', $remolques)
            ->update(['tractor_id' => $id, 'status' => 'Ocupado']);
    }

    // ✅ Assign selected dolly
    if (!empty($dolly_id)) {
        DB::table('dollys')
            ->where('id', $dolly_id)
            ->update(['tractor_id' => $id, 'status' => 'Ocupado']);
    }

    // 🔹 Gather full details for Slack notification
    $tractor = DB::table('tractores')->where('id', $id)->first();
    $conductor = $conductor_id ? DB::table('conductores')->where('id', $conductor_id)->first() : null;
    $remolquesList = !empty($remolques)
        ? DB::table('remolques')->whereIn('id', $remolques)->pluck('unidad')->toArray()
        : [];
    $dolly = $dolly_id ? DB::table('dollys')->where('id', $dolly_id)->first() : null;

    // 🧠 Format Slack message
    $text = "*🚛 Nueva Asignación de Tractor*\n\n"
        . "👨‍✈️ *Conductor:* " . ($conductor ? "{$conductor->first_name} {$conductor->last_name}" : "Ninguno") . "\n"
        . "🚜 *Tractor:* {$tractor->unidad} ({$tractor->brand} {$tractor->make})\n"
        . "🛻 *Remolque(s):* " . (!empty($remolquesList) ? implode(', ', $remolquesList) : "Ninguno") . "\n"
        . "🧲 *Dolly:* " . ($dolly ? $dolly->unidad : "Ninguno") . "\n\n"
        . "📅 *Fecha:* " . now()->format('d/m/Y H:i');

    // 🔸 Send message to Slack
    $webhookUrl = 'https://hooks.slack.com/services/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';

    Http::post($webhookUrl, [
        'text' => $text,
        'mrkdwn' => true,
    ]);

    return response()->json(['success' => true]);
}




}
